#region References

using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;
using Method = System.Reflection.MethodBase;
using System.Net;
using System.Text;
using System.Text.RegularExpressions;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7.OpenLibrary;

using ICTable = gov.va.med.vbecs.Common.VbecsTables.InterfaceControl;
using SProc = gov.va.med.vbecs.Common.VbecsStoredProcs;

// CR 2516 - Adding INTERFACE_TABLE reference due to InterfaceControl table being split into 2 tables, which creates AutoGen sync changes.. 
//		All code changes in this file can be found by searching for INTERFACE_TABLE
using INTERFACE_TABLE = gov.va.med.vbecs.Common.VbecsTables.Interface;

#endregion

namespace gov.va.med.vbecs.DAL.HL7AL
{
	#region Header

	//<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	//<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	//<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	//<Developers>
	//	<Developer>Brian    lin</Developer>
	//</Developers>
	//<SiteName>Hines OIFO</SiteName>
	//<CreationDate>5/10/2004</CreationDate>
	//<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	//<summary></summary>

	#endregion

	/// <summary>
	/// HL7Interface
	/// </summary>
	public class HL7Interface 
	{
		#region Variables

		private DataTable _structDataTable;
		private int _interfaceControlID;
		private string _interfaceName;
		private string _fieldSeparator;
		private char[] _encodingCharacters;
		private string _interfaceApplicationID;
		private string _vbecsApplicationID;
		private char _processingID;
		private string _versionID;
		private string _applicationAckType;
		
		private IPAddress _interfaceIPAddress;
		private string _interfaceDomainName;
		private int _interfacePortNumber;
		private string _interfaceFacilityID;
		
		private IPAddress _vbecsIPAddress;
		private int _vbecsPortNumber;
		private string _vbecsFacilityID;

		private int _ackTimeout;
		private byte _retransmitAttempts;
		private byte _purgeSuccessfulMessagesDays;
		private byte _purgeUnSuccessfulMessagesDays;
		private string _interfaceAdministratorEmail;

		private bool _logEvents;

		// CR 2940
		private bool _interfaceActiveIndicator;

		private bool _connectedTested;

		/// <summary>
		/// Used to notify registered listeners about interface parameter change. 
		/// </summary>
		public event EventHandler InterfaceParameterChanged;

		#endregion

		#region Constructors

		/// <summary>
		/// Constructor to initialize from <see cref="DataRow"/> object.
		/// </summary>
		/// <param name="dr"><see cref="DataRow"/> to get interface control parameters from.</param>
		private HL7Interface( DataRow dr ) 
		{
			_structDataTable = GetDataExchangeDataTableStructure();

			this.LoadFromDataRow( dr );
			OnInterfaceParameterChanged();
		}
 
		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4520"> 
		///		<ExpectedInput>Valid interface name.</ExpectedInput>
		///		<ExpectedOutput>Initialized object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4521"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Primary constructor to initialize from interface name.
		/// CR 2961
		/// </summary>
		/// <param name="interfaceName">Interface name</param>
		public HL7Interface( string interfaceName ) 
			: this( InterfaceControl.GetInterfaceControlParameters(interfaceName) ) { }

		#endregion

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/18/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8554"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8555"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2940
		/// </summary>
		public virtual bool ConnectedTested
		{
			get
			{
				return _connectedTested;
			}
			set
			{
				_connectedTested = value;
			}
		}

		///<Developers>
		///	<Developer>EDS EDS</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4339"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Non-empty list of interfaces.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4340"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets a list of all entries in the InterfaceControl table as <see cref="HL7Interface"/> objects.
		/// </summary>
		/// <returns><see cref="ArrayList"/>containing HL7Interface objects.</returns>
		public static IList GetHL7Interfaces()
		{
			ArrayList _list = new ArrayList();

			foreach( DataRow _dr in InterfaceControl.GetAllHL7Interfaces().Rows )
				_list.Add( new HL7Interface( _dr ) );

			return _list;
		}

		///<Developers>
		///	<Developer>Department of Veterans Affairs</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/26/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4522"> 
		///		<ExpectedInput>Valid app IDs combination</ExpectedInput>
		///		<ExpectedOutput>Non-null interface object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4524"> 
		///		<ExpectedInput>String, empty for both app IDs</ExpectedInput>
		///		<ExpectedOutput>DataAccessLayerException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4523"> 
		///		<ExpectedInput>interfaceApplicationId is null</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4525"> 
		///		<ExpectedInput>vbecsApplicationID is null</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 3082
		/// </summary>
		/// <param name="interfaceApplicationId"></param>
		/// <param name="vbecsApplicationId"></param>
		/// <returns></returns>
		public static HL7Interface GetInterfaceByApplicationIdentifiers( string interfaceApplicationId, string vbecsApplicationId )
		{
			if( interfaceApplicationId == null ) { throw( new ArgumentNullException("interfaceApplicationId") ); }
			if( vbecsApplicationId == null ) { throw( new ArgumentNullException("vbecsApplicationId") ); }
			//
			SqlParameter[] prms =
			{
				new SqlParameter( SProc.HL7GetInterfaceControlParameters.InterfaceApplicationId, SqlDbType.VarChar ),
				new SqlParameter( SProc.HL7GetInterfaceControlParameters.VbecsApplicationId, SqlDbType.VarChar )
			};
			//
			prms[0].Value = interfaceApplicationId;
			prms[1].Value = vbecsApplicationId;
			//
			DataRow interfaceRow = Common.StoredProcedure.GetSingleDataRow(SProc.HL7GetInterfaceControlParameters.StoredProcName, prms);
			if ( interfaceRow != null )
			{
				return new HL7Interface( interfaceRow );
			}
			//
			return null;
		}

		///<Developers>
		///	<Developer>Department of Veterans Affairs</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/15/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9239"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9240"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 3082
		/// </summary>
		/// <param name="interfaceControlId"></param>
		/// <returns></returns>
		public static HL7Interface GetInterfaceById( int interfaceControlId )
		{
			SqlParameter[] prms =
			{
				new SqlParameter( SProc.HL7GetInterfaceControlParameters.InterfaceControlId, SqlDbType.Int )
			};
			//
			prms[0].Value = interfaceControlId;
			//
			DataRow interfaceRow = Common.StoredProcedure.GetSingleDataRow(SProc.HL7GetInterfaceControlParameters.StoredProcName, prms);
			if ( interfaceRow != null )
			{
				return new HL7Interface( interfaceRow );
			}
			//
			return null;
		}

		/// <summary>
		/// Notifies registered listneres about interface parameter change 
		/// (used to perform validation of changed interface control parameters).
		/// </summary>
		protected void OnInterfaceParameterChanged()
		{
			if( InterfaceParameterChanged != null )
				InterfaceParameterChanged( this, EventArgs.Empty );
		}

		/// <summary>
		/// Validates supplied data row for presence of class required fields. 
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to check.</param>
		protected virtual void ValidateDataRow( DataRow dtRow )
		{
			if( dtRow == null )
				throw( new ArgumentNullException( "dtRow" ) );

			Utility.RequireColumns( dtRow, GetRequiredColumnsList() );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4528"> 
		///		<ExpectedInput>Valid DataRow.</ExpectedInput>
		///		<ExpectedOutput>Object is loaded with data from DataRow.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4529"> 
		///		<ExpectedInput>Null as DataRow.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4603"> 
		///		<ExpectedInput>Invalid DataRow.</ExpectedInput>
		///		<ExpectedOutput>ArgumentException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads data into this instance of the class from a supplied <see cref="DataRow"/>.
		/// CR 2940
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data from.</param>
		public virtual void LoadFromDataRow( DataRow dtRow )
		{
			ValidateDataRow( dtRow );

			_interfaceControlID = StDbNullConvert.ToInt32( dtRow[ ICTable.InterfaceControlId ] );
			_interfaceName = StDbNullConvert.ToString( dtRow[ INTERFACE_TABLE.InterfaceName ] );
			_fieldSeparator = StDbNullConvert.ToString( dtRow[ INTERFACE_TABLE.FieldSeparator ] );
			_encodingCharacters = StDbNullConvert.ToCharArray( dtRow[ INTERFACE_TABLE.EncodingCharacters ] );
			_interfaceApplicationID = StDbNullConvert.ToString( dtRow[ INTERFACE_TABLE.InterfaceApplicationId ] );
			_vbecsApplicationID = StDbNullConvert.ToString( dtRow[ INTERFACE_TABLE.VbecsApplicationId ] );
			_processingID = StDbNullConvert.ToChar( dtRow[ ICTable.ProcessingId ] );
			_versionID = StDbNullConvert.ToString( dtRow[ INTERFACE_TABLE.VersionId ] );
			_applicationAckType = StDbNullConvert.ToString( dtRow[ INTERFACE_TABLE.ApplicationAckType ] );
			
			_interfaceIPAddress = StDbNullConvert.ToIPAddress( dtRow[ ICTable.InterfaceIPAddress ] );
			_interfaceDomainName = StDbNullConvert.ToString( dtRow[ ICTable.InterfaceDomainName ] );
			_interfacePortNumber = StDbNullConvert.ToInt32( dtRow[ ICTable.InterfacePortNumber ] );
			_interfaceFacilityID = StDbNullConvert.ToString( dtRow[ ICTable.InterfaceFacilityId ] );

			_vbecsIPAddress = StDbNullConvert.ToIPAddress( dtRow[ ICTable.VbecsIPAddress ] );
			_vbecsPortNumber = StDbNullConvert.ToInt32( dtRow[ ICTable.VbecsPortNumber ] );
			_vbecsFacilityID = StDbNullConvert.ToString( dtRow[ ICTable.VbecsFacilityId ] );

			_ackTimeout = StDbNullConvert.ToInt32( dtRow[ ICTable.AckTimeout ] );
			_retransmitAttempts = StDbNullConvert.ToByte( dtRow[ ICTable.ReTransmitAttempts ] );
			_purgeSuccessfulMessagesDays = StDbNullConvert.ToByte( dtRow[ ICTable.PurgeSuccessfulMessage ] );
			_purgeUnSuccessfulMessagesDays = StDbNullConvert.ToByte( dtRow[ ICTable.PurgeUnSuccessfulMessage ] );
			_interfaceAdministratorEmail = StDbNullConvert.ToString( dtRow[ ICTable.InterfaceAdministratorName ] );

			_logEvents = StDbNullConvert.ToBool( dtRow[ ICTable.LogEventsIndicator ], false );

			// CR 2940
			_interfaceActiveIndicator = StDbNullConvert.ToBool( dtRow[ ICTable.InterfaceActiveIndicator ],  false);

			SetDefaultsForNonSpecifiedFields();
			OnInterfaceParameterChanged();
		}

		/// <summary>
		/// Set default interface parameters that were not specified. 
		/// </summary>
		protected virtual void SetDefaultsForNonSpecifiedFields()
		{
			if( StNullConvert.IsNull( _vbecsFacilityID ) )
				_vbecsFacilityID = "VBECS"; // MBR_4.01

			if( StNullConvert.IsNull( _purgeSuccessfulMessagesDays ) )
				_purgeSuccessfulMessagesDays = 7; // MBR_4.10

			if( StNullConvert.IsNull( _purgeUnSuccessfulMessagesDays ) )
				_purgeUnSuccessfulMessagesDays = 7; // MBR_4.10

			if( StNullConvert.IsNull( _ackTimeout ) ) 
				_ackTimeout = 10; // MBR_4.06

			if( StNullConvert.IsNull( _retransmitAttempts ) )
				_retransmitAttempts = 5; // MBR_4.07
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4530"> 
		///		<ExpectedInput>Valid DataRow.</ExpectedInput>
		///		<ExpectedOutput>DataRow is loaded with data from the object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4531"> 
		///		<ExpectedInput>Null as DataRow.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4602"> 
		///		<ExpectedInput>Invalid DataRow.</ExpectedInput>
		///		<ExpectedOutput>ArgumentException.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads data from this instance of the class into a supplied <see cref="DataRow"/>.
		/// CR 2940
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data into.</param>
		/// <returns>Reference to provided <see cref="DataRow"/>.</returns>
		public virtual DataRow LoadDataRowFromThis( DataRow dtRow )
		{
			ValidateDataRow( dtRow );

			dtRow[ ICTable.InterfaceControlId ] = StDbNullConvert.From( _interfaceControlID );
			dtRow[ INTERFACE_TABLE.InterfaceName ] = StDbNullConvert.From( _interfaceName );
			dtRow[ INTERFACE_TABLE.FieldSeparator ] = StDbNullConvert.From( _fieldSeparator );
			dtRow[ INTERFACE_TABLE.EncodingCharacters ] =  _encodingCharacters;
			dtRow[ INTERFACE_TABLE.InterfaceApplicationId ] = StDbNullConvert.From( _interfaceApplicationID );
			dtRow[ INTERFACE_TABLE.VbecsApplicationId ] = StDbNullConvert.From( _vbecsApplicationID );
			dtRow[ ICTable.ProcessingId ] = StDbNullConvert.From( _processingID );
			dtRow[ INTERFACE_TABLE.VersionId ] = StDbNullConvert.From( _versionID );
			dtRow[ INTERFACE_TABLE.ApplicationAckType ] = StDbNullConvert.From( _applicationAckType );
			
			dtRow[ ICTable.InterfaceIPAddress ] = StDbNullConvert.From( _interfaceIPAddress );
			dtRow[ ICTable.InterfaceDomainName ] = StDbNullConvert.From( _interfaceDomainName );
			dtRow[ ICTable.InterfacePortNumber ] = StDbNullConvert.From( _interfacePortNumber );
			dtRow[ ICTable.InterfaceFacilityId ] = StDbNullConvert.From( _interfaceFacilityID );

			dtRow[ ICTable.VbecsIPAddress ] = StDbNullConvert.From( _vbecsIPAddress );
			dtRow[ ICTable.VbecsPortNumber ] = StDbNullConvert.From( _vbecsPortNumber );
			dtRow[ ICTable.VbecsFacilityId ] = StDbNullConvert.From( _vbecsFacilityID );

			dtRow[ ICTable.AckTimeout ] = StDbNullConvert.From( _ackTimeout );
			dtRow[ ICTable.ReTransmitAttempts ] = StDbNullConvert.From( _retransmitAttempts );
			dtRow[ ICTable.PurgeSuccessfulMessage ] = StDbNullConvert.From( _purgeSuccessfulMessagesDays );
			dtRow[ ICTable.PurgeUnSuccessfulMessage ] = StDbNullConvert.From( _purgeUnSuccessfulMessagesDays );
			dtRow[ ICTable.InterfaceAdministratorName ] = StDbNullConvert.From( _interfaceAdministratorEmail );

			dtRow[ ICTable.LogEventsIndicator ] = StDbNullConvert.From( _logEvents );

			// CR 2940
			dtRow[ ICTable.InterfaceActiveIndicator ] = StDbNullConvert.From( _logEvents );

			return dtRow;
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/11/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7022"> 
		///		<ExpectedInput>Instance of HL7Interface for the CPRS interface.</ExpectedInput>
		///		<ExpectedOutput>DataRow with the InterfaceName column of 'CPRS'.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7023"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Exports an instance of the class as <see cref="DataRow"/>.
		/// </summary>
		/// <returns><see cref="DataRow"/> with data loaded from this instance of the class.</returns>
		public virtual DataRow ExportAsDataRow()
		{
			return LoadDataRowFromThis( _structDataTable.NewRow() );
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/18/2008</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8556"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8557"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// CR 2961
		/// </summary>
		/// <returns></returns>
		public virtual string GetInstanceData(bool criticalDataOnly)
		{
			StringBuilder instanceData = new StringBuilder();
			//
			instanceData.Append( "InterfaceControlID: ").Append(_interfaceControlID).Append("\n");
			instanceData.Append( "InterfaceName: ").Append(_interfaceName).Append("\n");
			instanceData.Append( "InterfaceIPAddress: ").Append(_interfaceIPAddress).Append("\n");
			instanceData.Append( "InterfaceDomainName: ").Append(_interfaceDomainName).Append("\n");
			instanceData.Append( "InterfacePortNumber: ").Append(_interfacePortNumber).Append("\n");
			//
			if ( !criticalDataOnly )
			{
				instanceData.Append( "FieldSeparator: ").Append(_fieldSeparator).Append("\n");
				instanceData.Append( "EncodingCharacters: ").Append(_encodingCharacters).Append("\n");
				instanceData.Append( "InterfaceApplicationID: ").Append(_interfaceApplicationID).Append("\n");
				instanceData.Append( "VbecsApplicationID: ").Append(_vbecsApplicationID).Append("\n");
				instanceData.Append( "ProcessingID: ").Append(_processingID).Append("\n");
				instanceData.Append( "VersionID: ").Append(_versionID).Append("\n");
				instanceData.Append( "ApplicationAckType: ").Append(_applicationAckType).Append("\n");
				instanceData.Append( "InterfaceFacilityID: ").Append(_interfaceFacilityID).Append("\n");
				instanceData.Append( "VbecsIPAddress: ").Append(_vbecsIPAddress).Append("\n");
				instanceData.Append( "VbecsPortNumber: ").Append(_vbecsPortNumber).Append("\n");
				instanceData.Append( "VbecsFacilityID: ").Append(_vbecsFacilityID).Append("\n");
				instanceData.Append( "AckTimeout: ").Append(_ackTimeout).Append("\n");
				instanceData.Append( "RetransmitAttempts: ").Append(_retransmitAttempts).Append("\n");
				instanceData.Append( "PurgeSuccessfulMessagesDays: ").Append(_purgeSuccessfulMessagesDays).Append("\n");
				instanceData.Append( "PurgeUnSuccessfulMessagesDays: ").Append(_purgeUnSuccessfulMessagesDays).Append("\n");
				instanceData.Append( "InterfaceAdministratorEmail: ").Append(_interfaceAdministratorEmail).Append("\n");
				instanceData.Append( "LogEvents: ").Append(_logEvents).Append("\n");
				instanceData.Append( "InterfaceActiveIndicator: ").Append(_interfaceActiveIndicator).Append("\n");
			}
			//
			return instanceData.ToString();
		}

		/// <summary>
		/// Gets a <see cref="DataTable"/> with columns needed to construct an 
		/// instance of the class from a row of such table.
		/// CR 2940
		/// </summary>
		/// <returns><see cref="DataTable"/> containing all required columns.</returns>
		protected virtual DataTable GetDataExchangeDataTableStructure()
		{
			DataTable _dt = new DataTable();

			_dt.Columns.Add( ICTable.InterfaceControlId, typeof(int) );
			_dt.Columns.Add( INTERFACE_TABLE.InterfaceName, typeof(string) );
			_dt.Columns.Add( INTERFACE_TABLE.FieldSeparator, typeof(string) );
			_dt.Columns.Add( INTERFACE_TABLE.EncodingCharacters, typeof(string) );
			_dt.Columns.Add( INTERFACE_TABLE.InterfaceApplicationId, typeof(string) );
			_dt.Columns.Add( INTERFACE_TABLE.VbecsApplicationId, typeof(string) );
			_dt.Columns.Add( ICTable.ProcessingId, typeof(char) );
			_dt.Columns.Add( INTERFACE_TABLE.VersionId, typeof(string) );
			_dt.Columns.Add( INTERFACE_TABLE.ApplicationAckType, typeof(string) );

			_dt.Columns.Add( ICTable.InterfaceIPAddress, typeof(string) );
			_dt.Columns.Add( ICTable.InterfaceDomainName, typeof(string) );
			_dt.Columns.Add( ICTable.InterfacePortNumber, typeof(int) );
			_dt.Columns.Add( ICTable.InterfaceFacilityId, typeof(string) );
			
			_dt.Columns.Add( ICTable.VbecsIPAddress, typeof(string) );
			_dt.Columns.Add( ICTable.VbecsPortNumber, typeof(int) );
			_dt.Columns.Add( ICTable.VbecsFacilityId, typeof(string) );

			_dt.Columns.Add( ICTable.AckTimeout, typeof(int) );
			_dt.Columns.Add( ICTable.ReTransmitAttempts, typeof(byte) );
			_dt.Columns.Add( ICTable.PurgeSuccessfulMessage, typeof(byte) );
			_dt.Columns.Add( ICTable.PurgeUnSuccessfulMessage, typeof(byte) );
			_dt.Columns.Add( ICTable.InterfaceAdministratorName, typeof(string) );

			_dt.Columns.Add( ICTable.LogEventsIndicator, typeof(bool) );

			// CR 2940
			_dt.Columns.Add( ICTable.InterfaceActiveIndicator, typeof(bool) );

			return _dt;
		}

		private string[] GetRequiredColumnsList()
		{
			string[] _requiredColumns = (string[])Array.CreateInstance( typeof(string), _structDataTable.Columns.Count );
			int nCount = 0;
			nCount = _structDataTable.Columns.Count;
			for( int i = 0; i < nCount; i++ )
				_requiredColumns[i] = _structDataTable.Columns[i].ColumnName;

			return _requiredColumns;
		}

		#region Properties

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4540"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid interface control ID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4541"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Interface control ID.
		/// </summary>
		public int InterfaceControlId
		{
			get
			{
				return _interfaceControlID;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4542"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid interface name.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4543"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Interface name.
		/// </summary>
		public string InterfaceName
		{
			get
			{
				return _interfaceName;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4544"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid field separator.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4545"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Message field separator.
		/// </summary>
		public string FieldSeparator
		{
			get
			{
				return _fieldSeparator;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4546"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid set of encoding characters.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4547"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Message encoding characters. 
		/// </summary>
		public char[] EncodingCharacters
		{
			get
			{
				return _encodingCharacters;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4548"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid interface application ID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4549"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Interface application ID.
		/// </summary>
		public string InterfaceApplicationId
		{
			get
			{
				return _interfaceApplicationID;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4550"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid VBECS application ID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4551"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VBECS application ID. 
		/// </summary>
		public string VbecsApplicationId
		{
			get
			{
				return _vbecsApplicationID;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4552"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid processing ID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4553"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Processing ID.
		/// </summary>
		public char ProcessingId
		{
			get
			{
				return _processingID;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4554"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid interface version ID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4555"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Version ID.
		/// </summary>
		public string VersionId
		{
			get
			{
				return _versionID;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4556"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid application ACK type.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4557"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Application ACK type. 
		/// </summary>
		public string ApplicationAckType
		{
			get
			{
				return _applicationAckType;
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4558"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid interface IP address.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4560"> 
		///		<ExpectedInput>Test IP address.</ExpectedInput>
		///		<ExpectedOutput>Matching test IP address.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4559"> 
		///		<ExpectedInput>Test IP address.</ExpectedInput>
		///		<ExpectedOutput>PropertyChangedEvent fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4561"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Interfaced application IP address. 
		/// </summary>
		public IPAddress InterfaceIPAddress
		{
			get
			{
				return _interfaceIPAddress;
			}
			set
			{
				if( value == _interfaceIPAddress )
					return;
				
				_interfaceIPAddress = value;
				OnInterfaceParameterChanged();
			}
		}

		///<Developers>
		///	<Developer>B.    in, C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/9/2007</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8494"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8495"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Interfaced application Domain Name.
		/// </summary>
		public string InterfaceDomainName
		{
			get
			{
				return _interfaceDomainName;
			}
			set
			{
				if( value == _interfaceDomainName )
					return;

				_interfaceDomainName = value;
				OnInterfaceParameterChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4562"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid interface port number.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4563"> 
		///		<ExpectedInput>Test interface port number.</ExpectedInput>
		///		<ExpectedOutput>Matching interface port number.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4564"> 
		///		<ExpectedInput>Test port number.</ExpectedInput>
		///		<ExpectedOutput>InterfaceParameterChanged fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4565"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Interfaced application port number. 
		/// </summary>
		public int InterfacePortNumber
		{
			get
			{
				return _interfacePortNumber;
			}
			set
			{
				if( value == _interfacePortNumber ) 
					return;

				_interfacePortNumber = value;
				OnInterfaceParameterChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4566"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid facility ID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4567"> 
		///		<ExpectedInput>Test facility ID.</ExpectedInput>
		///		<ExpectedOutput>Matching test facility ID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4568"> 
		///		<ExpectedInput>Test facility ID.</ExpectedInput>
		///		<ExpectedOutput>InterfaceParameterChanged fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4569"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Interfaced application facility ID.
		/// </summary>
		public string InterfaceFacilityId
		{
			get
			{
				return _interfaceFacilityID;
			}
			set
			{
				if( value == _interfaceFacilityID )
					return;

				_interfaceFacilityID = value;
				OnInterfaceParameterChanged();
			}
		}
		
		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4570"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid IP address.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4571"> 
		///		<ExpectedInput>Test IP address.</ExpectedInput>
		///		<ExpectedOutput>Matching test IP address.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4572"> 
		///		<ExpectedInput>Test IP address.</ExpectedInput>
		///		<ExpectedOutput>InterfaceParameterChanged fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4573"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VBECS side IP address. 
		/// </summary>
		public IPAddress VbecsIPAddress
		{
			get
			{
				return _vbecsIPAddress;
			}
			set
			{
				if( value == _vbecsIPAddress || ( value != null && value.Equals( _vbecsIPAddress ) ) )
					return;
				
				_vbecsIPAddress = value;
				OnInterfaceParameterChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4574"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid port number.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4575"> 
		///		<ExpectedInput>Test port number.</ExpectedInput>
		///		<ExpectedOutput>Matching test port number.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4576"> 
		///		<ExpectedInput>Test port number.</ExpectedInput>
		///		<ExpectedOutput>InterfaceParameterChanged fired</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4577"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VBECS side port nubmer. 
		/// </summary>
		public int VbecsPortNumber
		{
			get
			{
				return _vbecsPortNumber;
			}
			set
			{				 
				if( value == _vbecsPortNumber )
					return;

				_vbecsPortNumber = value;
				OnInterfaceParameterChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4578"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid facility ID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4579"> 
		///		<ExpectedInput>Test facility ID.</ExpectedInput>
		///		<ExpectedOutput>Matching test facility ID.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4580"> 
		///		<ExpectedInput>Test facility ID.</ExpectedInput>
		///		<ExpectedOutput>InterfaceParameterChanged fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4581"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// VBECS interface facility ID.
		/// </summary>
		public string VbecsFacilityId
		{
			get
			{
				return _vbecsFacilityID;
			}
			set
			{
				if( value == _vbecsFacilityID )
					return;

				_vbecsFacilityID = value;
				OnInterfaceParameterChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4582"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid ACK timeout.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4583"> 
		///		<ExpectedInput>Test ACK timeout.</ExpectedInput>
		///		<ExpectedOutput>Matching test ACK timeout.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4584"> 
		///		<ExpectedInput>Test ACK timeout.</ExpectedInput>
		///		<ExpectedOutput>InterfaceParameterChanged fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4585"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ACK timeout in seconds. 
		/// </summary>
		public int AckTimeout
		{
			get
			{
				if( _ackTimeout == int.MinValue )
				{
					return 5;
				}
				return _ackTimeout;
			}
			set
			{
				if( value == _ackTimeout )
					return;

				_ackTimeout = value;
				OnInterfaceParameterChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4586"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid number of retransmit attempts.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4587"> 
		///		<ExpectedInput>Test number of retransmit attempts.</ExpectedInput>
		///		<ExpectedOutput>Matching test number of retransmit attempts.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4588"> 
		///		<ExpectedInput>Test number of retransmit attempts.</ExpectedInput>
		///		<ExpectedOutput>InterfaceParameterChanged fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4589"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Number of retransmit attempts.
		/// </summary>
		public byte ReTransmitAttempts
		{
			get
			{
				return _retransmitAttempts;
			}
			set
			{
				if( value == _retransmitAttempts )
					return;

				_retransmitAttempts = value;
				OnInterfaceParameterChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4590"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid number of days to keep successful messages in queue.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4591"> 
		///		<ExpectedInput>Test number of days to keep successful messages in queue.</ExpectedInput>
		///		<ExpectedOutput>Matching test number of days to keep successful messages in queue.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4592"> 
		///		<ExpectedInput>Test number of days to keep successful messages in queue.</ExpectedInput>
		///		<ExpectedOutput>InterfaceParameterChanged fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4593"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Number of days to keep successfully processed messages in the database. 
		/// </summary>
		public byte PurgeSuccessfulMessage
		{
			get
			{
				return _purgeSuccessfulMessagesDays;
			}
			set
			{
				if( value == _purgeSuccessfulMessagesDays )
					return;

				_purgeSuccessfulMessagesDays = value;
				OnInterfaceParameterChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4594"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid number of days to keep unsuccessful messages in queue.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4595"> 
		///		<ExpectedInput>Test number of days to keep unsuccessful messages in queue.</ExpectedInput>
		///		<ExpectedOutput>Matching test number of days to keep unsuccessful messages in queue.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4596"> 
		///		<ExpectedInput>Test number of days to keep unsuccessful messages in queue.</ExpectedInput>
		///		<ExpectedOutput>InterfaceParameterChanged fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4597"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Number of days to keep erroneous messages in the database.
		/// </summary>
		public byte PurgeUnSuccessfulMessage
		{
			get
			{
				return _purgeUnSuccessfulMessagesDays;
			}
			set
			{
				if( value == _purgeUnSuccessfulMessagesDays )
					return;

				_purgeUnSuccessfulMessagesDays = value;
				OnInterfaceParameterChanged();
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/27/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4598"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid administrator e-mail.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4599"> 
		///		<ExpectedInput>Test administrator e-mail.</ExpectedInput>
		///		<ExpectedOutput>Matching test administrator e-mail.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="4600"> 
		///		<ExpectedInput>Test administrator e-mail.</ExpectedInput>
		///		<ExpectedOutput>InterfaceParameterChanged fired.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4601"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Interface administrator e-mail address. 
		/// </summary>
		public string InterfaceAdministratorEmail
		{
			get
			{
				return _interfaceAdministratorEmail;
			}
			set
			{
				if( value == _interfaceAdministratorEmail )
					return;

				_interfaceAdministratorEmail = value;
				OnInterfaceParameterChanged();
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8331"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Valid LogEvents indicator</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8332"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8339"> 
		///		<ExpectedInput>Test LogEvents indicator.</ExpectedInput>
		///		<ExpectedOutput>Matching LogEvents indicator.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8340"> 
		///		<ExpectedInput>Test LogEvents indicator.</ExpectedInput>
		///		<ExpectedOutput>InterfaceParameterChanged fired.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Log events indicator. 
		/// </summary>
		public bool LogEvents
		{
			get
			{
				return _logEvents;
			}
			set
			{
				if( value == _logEvents )
					return;

				_logEvents = value;
				OnInterfaceParameterChanged();
			}
		}

		/// <summary>
		/// Interface Active Indicator
		/// CR 2940
		/// </summary>
		public bool InterfaceActiveIndicator
		{
			get
			{
				return _interfaceActiveIndicator;
			}
			set
			{
				if (value == _interfaceActiveIndicator)
					return;

				_interfaceActiveIndicator = value;
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8333"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Hex value indicating End of Transmission.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8334"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// End-Of-Transmission (EOT) byte used to identify the end of an HL7 Message.
		/// </summary>
		public static byte EOTByte
		{
			get
			{
				return Convert.ToByte( 13 );
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8335"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Hex value indicating a Start of Transmission.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8336"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Start-Of-Transmission (SOT) byte used to identify the start of and HL7 Message.
		/// </summary>
		public static byte SOTByte
		{
			get
			{
				return Convert.ToByte( 11 );
			}
		}

		///<Developers>
		///	<Developer>Brian    lin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/29/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8337"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>Hex value indicating Field Separator.</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8338"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>None.</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// File Separator byte used to separate HL7 messages in a transmission.
		/// </summary>
		public static byte FSByte
		{
			get
			{
				return Convert.ToByte( 28 );
			}
		}



		#endregion
	}
}
 